{$R-}    {Range checking off}
{$B+}    {Boolean complete evaluation on}
{$S+}    {Stack checking on}
{$I+}    {I/O checking on}
{$N-}    {No numeric coprocessor}
{$M 65500,16384,655360} {Turbo 3 default stack and heap}

{6502 assembler by Andy Lieberman & John Haughey
 Copyright 1985, Caldetect, Inc.
 Last revised   4/8/85                                  }

Program assembler;


const
     IMM       =   0;
     REL       =  99;     {this offset shouldn't be added in}
     IMP       =  98;     {this offset shouldn't be added in}
     PGZ       = - 4;
     PGZ_X     =  12;
     PGZ_Y     =  12;
     ABS       =   4;
     ABS_X     =  20;
     ABS_Y     =  16;
     IND       =  36;
     IND_X     = - 8;
     IND_Y     =   8;

     ASL  = $0A;
     LSR  = $4A;
     ROL  = $2A;
     ROR  = $6A;

     label_exists              =  1;         {error codes}
     bad_label                 =  2;
     syntax_error              =  3;
     bad_operator              =  4;
     global_overflow           =  5;
     local_overflow            =  6;
     label_not_found           =  7;
     string_error              =  8;
     bad_branch_label          =  9;
     org_below_cur_address     = 10;
     branch_too_far            = 11;
     bad_dc_line               = 12;
     uneven_hex_string         = 13;
     not_a_hex_digit           = 14;
     bad_dc_i1                 = 15;
     too_many_local_replaces   = 16;
     too_many_forward_replaces = 17;

     max_globals        = 1000;
     max_symbols        = 200;
     max_need_replacing = 400;
     max_need_forward   = 800;
     number_of_opcodes  = 56;
     not_a_valid_opcode = $FF;

type
     line_of_code    = string[80];
     symbol          = record
                           name     : string[12];
                           address  : integer;
                       end;
     opcode          = record
                           name     : string[3];
                           value    : integer;
                       end;
     replace_data    = record
                           object_pointer : integer;
                           rep_label      : string[12];
                           addr_mode      : integer;
                           line_number    : integer;
                       end;

var
     opcode_data_file: text;
     cur_line        : line_of_code;
     temp_line       : line_of_code;
     out_line        : line_of_code;
     source_file     : text;
     global_symbols  : array[1..max_globals] of symbol;
     global_pointer  : integer;    {points to next global, number of globals is global_pointer - 1}
     object_file     : file of byte;
     asm_file        : text;
     file_name_1     : string[12];
     file_name_2     : string[12];
     file_name_3     : string[12];
     local_symbols   : array[1..max_symbols] of symbol;
     local_pointer   : integer;   {points to next local, number of locals is local_pointer - 1}
     need_replacing  : array[1..max_need_replacing] of replace_data;
     replace_pointer : integer;
     forward_replaces: array[1..max_need_forward] of replace_data;
     forward_pointer : integer;
     rep_label       : string[12];
     object_code     : array[0..$56ff] of byte;
     object_pointer  : integer;                         {from 0 to $7FFF}
     last_byte_of_obj: integer;           {length is this + 1}
     current_address : integer;
     start_address   : integer;           {first byte of obj. goes at this address}
     label_name      : string[40];
     operator        : integer;
     label_value     : integer;
     opcodes_file    : array[1..number_of_opcodes] of opcode;
     instruction     : string[40];
     opcode_value    : byte;
     addr_mode       : integer;
     list_on         : boolean;
     symbol_on       : boolean;
     print_on        : boolean;
     address         : integer;
     line_number     : integer;

procedure load_opcodes;
var
    counter      : integer;
    name         : string[3];
    value_string : string[4];
    value        : integer;
    result       : integer;

begin
    assign(opcode_data_file,'6502asm.dat');
    reset(opcode_data_file);
    for counter := 1 to number_of_opcodes do
    begin
        readln(opcode_data_file,name,value_string);
        opcodes_file[counter].name := name;
        value_string := copy(value_string,2,3);
        val(value_string,value,result);
        opcodes_file[counter].value := value;
    end;
    close(opcode_data_file);

end;

procedure initialize;
var
    y_or_n : string[1];

begin
    if paramcount < 1 then
    begin
      write('Enter source file name:');
      readln(file_name_1);
      write('Enter object file name:');
      readln(file_name_2);
      write('Enter assembly file name:');
      readln(file_name_3);
      write('Enter Y to save listing/symbols to disk: ');
      readln(y_or_n);
      if y_or_n = 'Y' then
        print_on := true
      else
        print_on := false;
      write('Enter Y to display/save symbols: ');
      readln(y_or_n);
      if y_or_n = 'Y' then
        symbol_on := true
      else
        symbol_on := false;
      write('Enter Y to display/save listing: ');
      readln(y_or_n);
      if y_or_n = 'Y' then
        list_on := true
      else
        list_on := false;
    end
    else
    begin
      file_name_1 := paramstr(1)+'.asm';
      file_name_2 := paramstr(1)+'.obj';
      file_name_3 := paramstr(1)+'.lst';
      print_on := true;
      symbol_on := true;
      list_on := true;
    end;
    assign(source_file,file_name_1);
    reset(source_file);
    assign(object_file,file_name_2);
    rewrite(object_file);
    assign(asm_file,file_name_3);
    rewrite(asm_file);
    local_pointer := 1;
    current_address := $0000;
    start_address := -1;
    object_pointer := 0;
    global_pointer :=1;
    temp_line := '';
    out_line := '';
    replace_pointer := 1;
    forward_pointer := 1;
    line_number := 1;
end;

procedure print_hex_byte(byte_value : byte);
var
   hex_digit : string[16];
   loop_var   : integer;
begin
   hex_digit := '0123456789ABCDEF';
   out_line := out_line + (hex_digit[(byte_value shr 4) +1]);
   out_line := out_line + (hex_digit[(byte_value and $F)+1]);
end;

procedure print_hex_integer(integer_value : integer);
begin
   print_hex_byte(hi(integer_value));
   print_hex_byte(lo(integer_value));
end;

procedure print_local;
var
   spaces   : integer;
   loop_var : integer;
begin
   if print_on then
   begin
       writeln(asm_file);
       writeln(asm_file,'Local symbol table.');
       writeln(asm_file);
   end
   else
   begin
       writeln;
       writeln('Local symbol table.');
       writeln;
   end;
   loop_var := 1;
   while loop_var <= local_pointer - 1 do
   begin
       out_line := local_symbols[loop_var].name;
       for spaces := 1 to (12 - length(local_symbols[loop_var].name)) do
           out_line := out_line + ' ';
       out_line := out_line + '$';
       print_hex_integer(local_symbols[loop_var].address);
       if print_on then
          writeln(asm_file,out_line)
       else
          writeln(out_line);
       loop_var := loop_var + 1;
   end;
   if print_on then
       writeln(asm_file)
   else
       writeln;
end;

procedure print_global;
var
   spaces   : integer;
   loop_var : integer;
begin
   if print_on then
   begin
       writeln(asm_file);
       writeln(asm_file,'Global symbol table.');
       writeln(asm_file);
   end
   else
   begin
       writeln;
       writeln('Global symbol table.');
       writeln;
   end;
   loop_var := 1;
   while loop_var <= global_pointer - 1 do
   begin
       out_line := global_symbols[loop_var].name;
       for spaces := 1 to (12 - length(global_symbols[loop_var].name)) do
           out_line := out_line + ' ';
       out_line := out_line + '$';
       print_hex_integer(global_symbols[loop_var].address);
       if print_on then
           writeln(asm_file,out_line)
       else
           writeln(out_line);
       loop_var := loop_var + 1;
   end;
end;

procedure error( error_code : integer);
var
   line_no : string[6];

begin
   str(line_number:5,line_no);
   line_no := line_no + ' ';
   if print_on then
   begin
   if list_on then
      writeln(asm_file,line_no,out_line,temp_line);
   case error_code of
   label_exists :
       writeln(asm_file,'Duplicate label.');
   bad_label :
       writeln(asm_file,'Label error.');
   syntax_error :
       writeln(asm_file,'Syntax error.');
   bad_operator :
       writeln(asm_file,'Operator error.');
   global_overflow :
       writeln(asm_file,'Global label overflow.');
   local_overflow :
       writeln(asm_file,'Local label overflow.');
   label_not_found :
       writeln(asm_file,'Label ',label_name,' not found.');
   string_error :
       writeln(asm_file,'String error.');
   bad_branch_label :
       writeln(asm_file,'Bad branch label.');
   org_below_cur_address :
       writeln(asm_file,'ORG below current address.');
   branch_too_far :
       writeln(asm_file,'Branch too far.');
   bad_dc_line :
       writeln(asm_file,'Bad DC command.');
   uneven_hex_string :
       writeln(asm_file,'Hex string contains an uneven number of digits.');
   not_a_hex_digit :
       writeln(asm_file,'Not a valid hex digit.');
   bad_dc_i1 :
       writeln(asm_file,'I1 value > $FF.');
   too_many_local_replaces :
       writeln(asm_file,'Too many local replaces.');
   too_many_forward_replaces :
       writeln(asm_file,'Too many forward replaces. ');
   end;
   end;
   if list_on then
      writeln(out_line,line_no,temp_line);
   case error_code of
   label_exists :
       writeln('Duplicate label.');
   bad_label :
       writeln('Label error.');
   syntax_error :
       writeln('Syntax error.');
   bad_operator :
       writeln('Operator error.');
   global_overflow :
       writeln('Global label overflow.');
   local_overflow :
       writeln('Local label overflow.');
   label_not_found :
       writeln('Label ',label_name,' not found.');
   string_error :
       writeln('String error.');
   bad_branch_label :
       writeln('Bad branch label.');
   org_below_cur_address :
       writeln('ORG below current address.');
   branch_too_far :
       writeln('Branch too far.');
   bad_dc_line :
       writeln('Bad DC command.');
   uneven_hex_string :
       writeln('Uneven number of digits in hex string.');
   not_a_hex_digit :
       writeln('Not a valid hex digit.');
   bad_dc_i1 :
       writeln('I1 value > $FF.');
   too_many_local_replaces :
       writeln('Too many local replaces.');
   too_many_forward_replaces :
       writeln('Too many forward replaces.');
   end;
   writeln('line_number : ',line_number);
   writeln('cur_line : ',cur_line);
   writeln('operator : ',operator);
   writeln('temp_line : ',temp_line);
   writeln('label_name : ',label_name);
   writeln('instruction : ',instruction);
   writeln('file name : ',file_name_1);
   close(asm_file);
   halt;
end;

function hex_to_dec(hex_char : char) :integer;
var
   dec : integer;
begin
   dec := ord(hex_char);
   if (dec >= 48) and (dec <= 57) then
       hex_to_dec := dec - 48
   else
   if (dec >= 65) and (dec <= 70) then
       hex_to_dec := dec - 55
   else
   if (dec >= 97) and (dec <= 102) then
       hex_to_dec := dec - 87
   else
       error(not_a_hex_digit);

end;

procedure clear_all_blanks;
begin
    while (cur_line[1] = ' ') and (length(cur_line) > 0) do
       cur_line := copy(cur_line,2,length(cur_line)-1);
    if cur_line = ' ' then
       cur_line := '';

end;

function read_line:boolean;
var
   pointer   : integer;
   number    : string[6];

begin
   str(line_number:5,number);
   number := number + ' ';
   if print_on and list_on then
      writeln(asm_file, number, out_line, temp_line)
   else if not(print_on) and list_on then
      writeln(number, out_line, temp_line);
   line_number := line_number + 1;
   out_line := '';
   temp_line := '';
   read_line := not(eof(source_file));
   if not(eof(source_file)) then
   begin
      readln(source_file,cur_line);
      temp_line := cur_line;
      cur_line := cur_line + ' ;';
      pointer := 1;
      while (pointer <= length(cur_line)) and (cur_line[pointer] <> '''') and
            (cur_line[pointer] <> '"') do
            begin
               cur_line[pointer] := upcase(cur_line[pointer]);
               pointer := pointer + 1;
           end;
   end;
end;

function comment:boolean;
begin
     if (cur_line[1] = ';') or ((cur_line[1] <> ' ') and (cur_line[1] <> '.')
          and ((cur_line[1] < 'A') or (cur_line[1] > 'Z'))) or
          (length(cur_line) < 3) or (cur_line  = '') then
     begin
         comment:=true;
         out_line := out_line + '              ';
     end
     else
         comment:=false;

end;

function is_it_global : boolean;
begin
    if (copy(cur_line,1,4) = 'GEQU') or (copy(cur_line,1,5)='ENTRY') or
       (copy(cur_line,1,5) = 'START') or (copy(cur_line,1,3)='ORG') then
        begin
          is_it_global := true;
{          writeln(temp_line);}
        end
    else
          is_it_global := false;

end;

function check_for_x : boolean;
var
   loop_var : integer;
begin
   check_for_x := false;
   loop_var := 1;
   while (loop_var < length(cur_line)) and (cur_line[loop_var] <> ';') do
   begin
       if (cur_line[loop_var] = ',') and (cur_line[loop_var+1] = 'X') then
          check_for_x := true
       else
       if (cur_line[loop_var] = ',') and (cur_line[loop_var+1] = 'Y') then
          check_for_x := false
       else
       if (cur_line[loop_var] = ',') then
          error(bad_label);
       loop_var := loop_var + 1;
   end;
end;

function check_for_y: boolean;
var
   loop_var : integer;
begin
   check_for_y := false;
   loop_var := 1;
   while(loop_var < length(cur_line)) and (cur_line[loop_var] <> ';') do
   begin
       if (cur_line[loop_var] = ',') and (cur_line[loop_var+1] = 'Y') then
          check_for_y := true
       else
       if (cur_line[loop_var] = ',') and (cur_line[loop_var+1] = 'X') then
          check_for_y := false
       else
       if (cur_line[loop_var] = ',') then
          error(bad_label);
       loop_var := loop_var + 1;
   end;
end;

function find_label(var label_value :integer):boolean;
var
   loop_var    : integer;
   search_done : boolean;
   lo_or_hi    : string[1];
   symbol      : char;
   number      : integer;
   result      : integer;
   temp_label  : string[24];
   operand     : string[12];
   found_it    : boolean;
begin
       temp_label := label_name;
       symbol := '?';
       found_it := false;
       for loop_var := 1 to length(label_name) do
          if (label_name[loop_var] = '+') or (label_name[loop_var] = '-') or
             (label_name[loop_var] = '*') or (label_name[loop_var] = '/') then
          begin
              symbol := label_name[loop_var];
              val(copy(label_name,loop_var+1,9999),number,result);
              label_name := copy(label_name,1,loop_var-1);
          end;
       if (label_name[1] = '>') or (label_name[1] = '<') then
       begin
          lo_or_hi := label_name[1];
          label_name := copy(label_name,2,length(label_name)-1);
       end
       else
          lo_or_hi := '';
       find_label  := false;
       found_it := false;
       search_done := false;
       if label_name = 'A' then
       begin
          find_label := true;
          found_it := false;
          search_done := true;
       end;
       loop_var := 1;
       while loop_var <= local_pointer - 1 do
       begin
          if local_symbols[loop_var].name = label_name then
          begin
             search_done := true;
             find_label  := true;
             found_it    := true;
             label_value := local_symbols[loop_var].address;
             if lo_or_hi = '>' then
                 label_value := hi(label_value)
             else
             if lo_or_hi = '<' then
                 label_value := lo(label_value);
             loop_var    := local_pointer;
          end;
          loop_var := loop_var + 1;
       end;
       if not(search_done) then
       begin
          loop_var := global_pointer - 1;
          while loop_var >= 1 do
          begin
              if global_symbols[loop_var].name = label_name then
              begin
                find_label  := true;
                found_it    := true;
                label_value := global_symbols[loop_var].address;
                if lo_or_hi = '>' then
                    label_value := hi(label_value)
                else
                if lo_or_hi = '<' then
                    label_value := lo(label_value);
                loop_var := 1;
              end;
              loop_var := loop_var - 1;
          end;
       end;
       if lo_or_hi <> '' then
           label_name := lo_or_hi + label_name;
       if found_it = true then
           if symbol <> '?' then
           begin
              case symbol of
                 '+' : label_value := label_value + number;
                 '-' : label_value := label_value - number;
                 '*' : label_value := label_value * number;
                 '/' : label_value := label_value div number;
              end;
           end;
       label_name := temp_label;
end;

procedure translate_op;
var
   temp_label : string[12];
   temp_storage : line_of_code;
   symbol       : char;
   number       : integer;
   temp_line    : line_of_code;
   loop_var     : integer;

begin
   temp_storage := cur_line;
   temp_label := label_name;
   label_name := '';
   while (length(cur_line)<>0) and (cur_line[1]<>')') and (cur_line[1]<>' ')
      and (cur_line[1]<>',') do
   begin
      label_name := label_name + cur_line[1];
      if length(cur_line) > 1 then
          cur_line := copy(cur_line,2,length(cur_line)-1)
      else
          cur_line := '';
   end;
   if (label_name <> 'A') and find_label(operator) then
   begin
      if (cur_line[1] <> ')') and (cur_line[1] <> ' ') and (cur_line[1] <> ',') and
         (length(cur_line) <> 0) then
          begin
             symbol := cur_line[1];
             cur_line := copy(cur_line,2,length(cur_line)-1);
             clear_all_blanks;
             temp_line := '';
             loop_var := 1;
             while (cur_line[loop_var] <> ' ') and (cur_line[loop_var] <> ';') and
                 (loop_var <= length(cur_line)) do
             begin
                temp_line := temp_line + cur_line[loop_var];
                loop_var := loop_var + 1;
             end;
             val(temp_line,number,loop_var);
             case symbol of
             '+' :
                 operator := operator + number;
             '-' :
                 operator := operator - number;
             '*' :
                 operator := operator * number;
             '/' :
                 operator := operator div number;
             else
                 error(bad_operator);
             end;
          end;
   end
   else
   begin
      operator := $7FFF;
      rep_label := label_name;    {save any LABEL+1 type labels}
   end;
   label_name := temp_label;
   cur_line := temp_storage;
end;

procedure ascii_input;
begin
   if ((cur_line[1] = '''') and (cur_line[3] = '''')) or
      ((cur_line[1] = '"') and (cur_line[3] = '"')) then
   begin
       operator := ord(cur_line[2]);
       cur_line := '';
   end
   else
      error(string_error);

end;

procedure compute_operator;
var
   loop_var : integer;
   temp_line : string[80];
   result : integer;
   counter : integer;
begin
   operator := 0;
   temp_line:='';
   loop_var := 1;
   clear_all_blanks;
   if length(cur_line) = 1 then
      cur_line := cur_line + ';';
   if (cur_line = '') or (cur_line[1] = ';') or (((opcode_value = ASL) or
      (opcode_value = LSR) or (opcode_value = ROR) or (opcode_value = ROL)) and
       (cur_line[1] = 'A') and ((cur_line[2] < 'A') or (cur_line[2] > 'Z'))) then
        addr_mode := IMP
   else if (instruction[1] = 'B') and (instruction <> 'BIT') and
           (instruction <> 'BRK') then
           begin
              addr_mode := REL;
              operator := $7FFF;
              counter := 1;
              while (cur_line[counter] <> ' ') and (cur_line[counter] <> ';') and
                    (counter < length(cur_line)) do
                        counter := counter + 1;
              rep_label := copy(cur_line,1,counter);
              if rep_label[length(rep_label)] = ' ' then
                  rep_label := copy(rep_label,1,length(rep_label) - 1);
              cur_line := '';
           end
   else if cur_line[1] = '(' then
        begin
        if check_for_x then
           addr_mode := IND_X
        else
        if check_for_y then
           addr_mode := IND_Y
        else
           addr_mode := IND;  {for jump}
        cur_line := copy(cur_line,2,length(cur_line)-1);
        translate_op;
   end
   else if cur_line[1] = '#' then
   begin
      addr_mode := IMM;
      cur_line := copy(cur_line,2,length(cur_line) - 1);
      if ((cur_line[1] >= 'A') and (cur_line[1] <= 'Z')) or (cur_line[1] = '>')
         or (cur_line[1] = '<') then
         translate_op
      else
      if (cur_line[1] = '''') or (cur_line[1] = '"') then
         ascii_input
      else
        while (cur_line[loop_var] <> ' ') and (loop_var <= length(cur_line)) do
        begin
            temp_line := temp_line + cur_line[loop_var];
            loop_var := loop_var + 1;
            val(temp_line,operator,result);
        end;
   end
   else if (cur_line[1] >= 'A') and (cur_line[1] <= 'Z') then
   begin
        translate_op;
        if (((operator > $FF) or (operator < $0)) and (instruction <> 'GEQU')) or
           (instruction = 'JMP') or (instruction = 'JSR') then
        begin
            if check_for_x then
               addr_mode := ABS_X
            else
            if check_for_y then
               addr_mode := ABS_Y
            else
               addr_mode := ABS;
        end
        else
        if instruction <> 'GEQU' then
        begin
            if check_for_x then
               addr_mode := PGZ_X
            else
            if check_for_y and (instruction <> 'LDX') then
               addr_mode := ABS_Y
            else
            if check_for_y and (instruction = 'LDX') then
               addr_mode := PGZ_Y
            else
               addr_mode := PGZ ;
        end;
   end
   else if (cur_line[1] = '$') or ((cur_line[1] >= '0') and (cur_line[1] <= '9')) then
   begin
        loop_var := 1;
        while (cur_line[loop_var] <> ' ') and (cur_line[loop_var] <> ',')
            and (loop_var <= length(cur_line)) do
        begin
            temp_line := temp_line + cur_line[loop_var];
            loop_var := loop_var + 1;
        end;
        val(temp_line,operator,result);
        if loop_var < length(cur_line) then
           cur_line := copy(cur_line,loop_var,length(cur_line)-loop_var+1)
        else
           cur_line := '';
        if (operator <= $FF) and (operator >= $0) then
        begin
           if check_for_x then
              addr_mode := PGZ_X
           else
           if check_for_y and (instruction = 'LDX') then
              addr_mode := PGZ_Y
           else
           if check_for_y and (instruction <> 'LDX') then
              addr_mode := ABS_Y
           else
              addr_mode := PGZ;
        end
        else
        begin
           if check_for_x then
              addr_mode := ABS_X
           else
           if check_for_y then
              addr_mode := ABS_Y
           else
              addr_mode := ABS;
       end;
   end
   else error(bad_operator);
end;

procedure interp_global; { and store it to the symbol file. }
var
   counter : integer;
begin
       if global_pointer > max_globals then
          error(global_overflow);
       global_symbols[global_pointer].name:= label_name;
       if (copy(cur_line,1,4) = 'GEQU') then
       begin
          cur_line := copy(cur_line,5,length(cur_line)-4);
          clear_all_blanks;
          instruction := 'GEQU';
          compute_operator; { 1+2,addr1-20,(addr),x,etc. }
          global_symbols[global_pointer].address := operator;
       end
       else
       if (copy(cur_line,1,3) = 'ORG') then
       begin
          cur_line := copy(cur_line,4,length(cur_line)-3);
          clear_all_blanks;
          instruction := 'ORG';
          compute_operator; { 1+2,addr1-20,(addr),x,etc. }
          if start_address = -1 then
              start_address := operator;
          if (current_address <> 0) and (operator < current_address) then
              error(org_below_cur_address);
          if (current_address <> 0) and ((operator - 1)>current_address) then
             for counter := current_address to (operator - 1) do
             begin
                object_code[object_pointer] := 0;
                object_pointer := object_pointer + 1;
             end;
          current_address := operator;
          global_symbols[global_pointer].address := operator;
       end
       else
          global_symbols[global_pointer].address := current_address;
       if find_label(label_value) then
          error(label_exists);
       global_pointer := global_pointer + 1;
end;

procedure extract_label;
var
    end_pointer : integer;

begin
    clear_all_blanks;
    end_pointer := 1;
    while (cur_line[end_pointer] <> ' ') and (end_pointer <= length(cur_line)) do
        if end_pointer < 12 then
             end_pointer := end_pointer + 1
        else
             error(bad_label);
    label_name := copy(cur_line,1,end_pointer-1);
    if cur_line = label_name then
        cur_line := ';'
    else
        cur_line := copy(cur_line,end_pointer,length(cur_line)-end_pointer+1);
    clear_all_blanks;
    if is_it_global then
    begin
       interp_global;
       cur_line := ';';
    end
    else
    begin
        if local_pointer > max_symbols then
           error(local_overflow);
        local_symbols[local_pointer].name := label_name;
        local_symbols[local_pointer].address := current_address;
        local_pointer := local_pointer + 1;
    end;
end;

procedure check_label;
begin
    if (cur_line[1]<>';') and (cur_line[1]<>' ') and (length(cur_line) > 3) then
        extract_label; {and put it in the symbol table}
end;

procedure get_opcode;
var
    counter : integer;

begin
    counter := 1;
    opcode_value := not_a_valid_opcode;
    while (counter <= number_of_opcodes) and (opcode_value = not_a_valid_opcode) do
    begin
        if instruction = opcodes_file[counter].name then
            opcode_value := opcodes_file[counter].value;
        counter := counter + 1;
    end;

end;

procedure write_object_bytes;
begin
    print_hex_integer(current_address);
    out_line := out_line + ' ';
    object_code[object_pointer] := opcode_value;
    print_hex_byte(opcode_value);
    out_line := out_line + ' ';
    current_address := current_address + 1;
    object_pointer := object_pointer + 1;
    if (addr_mode=PGZ) or (addr_mode=PGZ_X) or (addr_mode=PGZ_Y) or
       (addr_mode=IND_X) or (addr_mode=IND_Y) or
       (addr_mode=REL) or (addr_mode=IMM) then
    begin
        object_code[object_pointer] := operator;
        print_hex_byte(operator);
        out_line := out_line + '    ';
        current_address := current_address + 1;
        object_pointer := object_pointer + 1;
    end
    else
    if (addr_mode=ABS) or (addr_mode=ABS_X) or (addr_mode=ABS_Y) or
       (addr_mode=IND) then
    begin
        object_code[object_pointer] := lo(operator) {low byte};
        object_code[object_pointer + 1] := hi(operator); {hi byte};
        print_hex_byte(lo(operator));
        out_line := out_line + ' ';
        print_hex_byte(hi(operator));
        out_line := out_line + ' ';
        current_address := current_address + 2;
        object_pointer := object_pointer + 2;
    end
    else
        out_line := out_line + '      ';
end;

procedure check_for_exceptions;
begin
    if ((instruction = 'CPX') or (instruction = 'CPY') or
        (instruction = 'LDX') or (instruction = 'LDY')) and
       (addr_mode = IMM) then
           opcode_value := opcode_value - 8;

end;

function valid_opcode : boolean;
begin
    get_opcode;
    if opcode_value = not_a_valid_opcode then
        valid_opcode := false
    else
    begin
        valid_opcode := true;
        compute_operator;
        check_for_exceptions;
        if (addr_mode<>REL) and (addr_mode<>IMP) then
             opcode_value := opcode_value + addr_mode;
        if operator = $7FFF then
        begin
            if replace_pointer > max_need_replacing then
                error(too_many_local_replaces);
            need_replacing[replace_pointer].object_pointer := object_pointer + 1;
            need_replacing[replace_pointer].rep_label := rep_label;
            need_replacing[replace_pointer].addr_mode := addr_mode;
            need_replacing[replace_pointer].line_number := line_number;
            replace_pointer := replace_pointer + 1;
        end;
        write_object_bytes;
    end;

end;

procedure title;
begin
end;

procedure end_of_source;
begin
    while not(eof(source_file)) do     {move pointer to end of file}
       readln(source_file,cur_line);
end;

procedure pass_two; forward;

procedure apend;
var
    counter : integer;
begin
    pass_two;
    if symbol_on then
       print_local;
    local_pointer := 1;
    counter := 1;
    while (cur_line[counter]<>' ') and (counter <= length(cur_line)) do
        counter := counter + 1;
    file_name_1 := copy(cur_line,1,counter);
    if file_name_1[length(file_name_1)] = ' ' then
        file_name_1 := copy(file_name_1,1,length(file_name_1)-1);
    if print_on then
       writeln(asm_file,'Appending ',file_name_1);
    writeln('Appending ',file_name_1);
    close(source_file);
    assign(source_file,file_name_1);
    reset(source_file);
    cur_line := '';
    temp_line := '';
    out_line := '';
end;

procedure dc_i1;
var
    value : integer;
begin
    label_name := cur_line;
    if find_label(value) then
    begin
        if (value < $00) or (value > $FF) then
            error(bad_dc_i1);
        object_code[object_pointer] := value;
        object_pointer := object_pointer + 1;
        current_address := current_address + 1;
    end
    else
        error(label_not_found);
end;

procedure dc_h;
var
    hex_byte_string : string[2];
    hex_byte        : integer;      {number from 0 to 255}
begin
    if (length(cur_line) mod 2)<>0 then
        error(uneven_hex_string);
    while length(cur_line) >= 2 do
    begin
        hex_byte_string := copy(cur_line,1,2);
        cur_line := copy(cur_line,3,length(cur_line)-2);
        hex_byte := $10 * hex_to_dec(hex_byte_string[1]) + hex_to_dec(hex_byte_string[2]);
        object_code[object_pointer] := hex_byte;
        object_pointer := object_pointer + 1;
        current_address := current_address + 1;
    end;
end;

procedure dc_a;
var
    value : integer;
begin
    label_name := cur_line;
    if find_label(value) then
    begin
        object_code[object_pointer] := lo(value);
        object_code[object_pointer + 1] := hi(value);
        object_pointer := object_pointer + 2;
        current_address := current_address + 2;
   end
   else
   begin
       if replace_pointer > max_need_replacing then
            error(too_many_local_replaces);
       need_replacing[replace_pointer].object_pointer := object_pointer;
       object_pointer := object_pointer + 2;
       current_address := current_address + 2;
       need_replacing[replace_pointer].rep_label := label_name;
       need_replacing[replace_pointer].addr_mode := ABS;
       need_replacing[replace_pointer].line_number := line_number;
       replace_pointer := replace_pointer + 1;
   end;
end;

procedure dc_c;
begin
    while cur_line <> '' do
    begin
        object_code[object_pointer] := ord(cur_line[1]);
        object_pointer := object_pointer + 1;
        current_address := current_address + 1;
        if length(cur_line) = 1 then
            cur_line := ''
        else
            cur_line := copy(cur_line,2,length(cur_line)-1);
    end;
end;

procedure dc;
var
    counter : integer;
    opening_quote : string[1];
    dc_type : line_of_code;
begin
    out_line := '';
    print_hex_integer(current_address);
    out_line := out_line + '          ';
    counter := 1;
    while (cur_line[counter] <> '''') and (cur_line[counter] <> '"') and
        (counter < length(cur_line)) do
        counter := counter + 1;
   if counter = length(cur_line) then
        error(bad_dc_line);
   dc_type := copy(cur_line,1,counter - 1);
   opening_quote := cur_line[counter];
   if (opening_quote <> '''') and (opening_quote <> '"') then
       error(bad_dc_line);
   cur_line := copy(cur_line,counter+1,999);
   counter := 1;
   while (cur_line[counter] <> opening_quote) and (counter < length(cur_line)) do
       counter := counter + 1;
   if cur_line[counter] <> opening_quote then
       error(bad_dc_line);
   cur_line := copy(cur_line,1,counter - 1);
   if dc_type = 'I1' then
       dc_i1
   else if dc_type = 'H' then
       dc_h
   else if dc_type = 'A' then
       dc_a
   else if dc_type = 'C' then
       dc_c
   else
       error(bad_dc_line);
end;

procedure anop;
begin
    cur_line := ''
end;

procedure act_on_directive;
begin
    if instruction = 'ANOP' then
        anop
    else if instruction = 'DC' then
        dc
    else if instruction = 'APPEND' then
        apend
    else if instruction = 'END' then
        end_of_source
    else if instruction = 'TITLE' then
        title
    else if instruction = 'PRNTON' then
        print_on := true
    else if instruction = 'PRNTOFF' then
        print_on := false
    else if instruction = 'SYMBOLON' then
        symbol_on := true
    else if instruction = 'SYMBOLOFF' then
        symbol_on := false
    else if instruction = 'LISTON' then
        list_on := true
    else if instruction = 'LISTOFF' then
        list_on := false
    else
       error(syntax_error);
end;

procedure do_directive;
begin
    clear_all_blanks;
    out_line := '              ';
    act_on_directive;
end;

procedure translate;
var
   end_pointer : integer;

begin
    end_pointer := 1;
    while (cur_line[end_pointer] <> ' ') and (end_pointer < length(cur_line)) do
       end_pointer := end_pointer + 1;
    instruction := copy(cur_line,1,end_pointer);
    if instruction[length(instruction)]=' ' then
       instruction := copy(instruction,1,length(instruction)-1);
    if length(cur_line) > length(instruction) then
       cur_line := copy(cur_line,end_pointer,length(cur_line)-end_pointer+1)
    else
       cur_line := '';
    clear_all_blanks;
    if not(valid_opcode) then
           do_directive;
end;

procedure pass_one;
begin
   writeln('Pass one.');
   while read_line do
   begin
      check_label;
      clear_all_blanks;
      if not(comment) then
             translate;
   end;
   if symbol_on then
   begin
      print_local;
      print_global;
   end;
end;

procedure replace_relative(object_loc,address : integer);
var
    branch_dist : integer;
begin
    branch_dist := address - (start_address + object_loc);
    if (branch_dist >= 1) and (branch_dist <= $80) then
        object_code[object_loc] := branch_dist - 1
    else
    if (branch_dist <= 0) and (branch_dist >= -$7F) then
        object_code[object_loc] := $FF + branch_dist
    else
    begin
        writeln('Branch dist : ',branch_dist);
        writeln('start_address : ',start_address);
        writeln('address : ',address);
        writeln('object_loc : ',object_loc);
        error(branch_too_far);
   end;
end;

procedure replace_page_zero(object_loc,address : integer);
begin
    object_code[object_loc] := address;
end;

procedure replace_two_bytes(object_loc,address : integer);
begin
    object_code[object_loc] := lo(address);
    object_code[object_loc + 1] := hi(address);
end;

procedure replace_label(object_loc: integer; label_name: line_of_code;
                        mode: integer; line_no: integer);
begin
    if find_label(address) then
    begin
        if mode = REL then
            replace_relative(object_loc,address)
        else
        if (mode=ABS) or (mode=ABS_X) or (mode=ABS_Y) or (mode=IND) then
           replace_two_bytes(object_loc,address)
        else
           replace_page_zero(object_loc,address);
    end
    else
    begin
        if forward_pointer > max_need_forward then
            error(too_many_forward_replaces);
        forward_replaces[forward_pointer].object_pointer := object_loc;
        forward_replaces[forward_pointer].rep_label := label_name;
        forward_replaces[forward_pointer].addr_mode := addr_mode;
        forward_replaces[forward_pointer].line_number := line_no;
        forward_pointer := forward_pointer + 1;
    end;
end;

procedure pass_two;
var
   number_of_replaces  : integer;
   addr_goes_at : integer;
   counter : integer;
   line_no : integer;

begin
   number_of_replaces := replace_pointer - 1;
   counter := 1;
   while counter <= number_of_replaces do
     begin
       addr_goes_at := need_replacing[counter].object_pointer;
       label_name := need_replacing[counter].rep_label;
       addr_mode := need_replacing[counter].addr_mode;
       line_no := need_replacing[counter].line_number;
       replace_label(addr_goes_at,label_name,addr_mode,line_no);
       counter := counter + 1;
     end;
   replace_pointer := 1;
end;

procedure forward_global_replacements;
var
    counter : integer;
begin
  local_pointer := 1;   {These can't be local equates, or they would}
  counter := 1;           {have already been replaced}
  while counter <= (forward_pointer - 1) do
  begin
    object_pointer := forward_replaces[counter].object_pointer;
    label_name     := forward_replaces[counter].rep_label;
    addr_mode      := forward_replaces[counter].addr_mode;
    line_number    := forward_replaces[counter].line_number;
    if find_label(address) then
    begin
        if addr_mode = REL then
            replace_relative(object_pointer,address)
        else
        if (addr_mode=ABS) or (addr_mode=ABS_X) or (addr_mode=ABS_Y)
           or (addr_mode=IND) then
           replace_two_bytes(object_pointer,address)
        else
           replace_page_zero(object_pointer,address);
    end
    else
        error(label_not_found);
    counter := counter + 1;
  end;
end;

procedure save_obj_code;
var
   counter : integer;
begin
  for counter := 0 to last_byte_of_obj do
      write(object_file,object_code[counter]);
  close(object_file);
  close(asm_file);
  writeln;
  writeln('Successfully wrote ',last_byte_of_obj + 1,' bytes of object code.');
end;

begin { Main Program. }
    load_opcodes;
    initialize;
    pass_one;
    last_byte_of_obj := object_pointer - 1;
    out_line := '';
    temp_line := '';
    cur_line := '';
    pass_two;
    writeln('Linking...');
    forward_global_replacements;
    save_obj_code;
end.
